const jwt = require('jsonwebtoken');
const pool = require('../config/database');

const authMiddleware = async (req, res, next) => {
    try {
        // Get token from header
        const authHeader = req.headers.authorization;
        if (!authHeader) {
            return res.status(401).json({
                success: false,
                error: 'No authorization token found'
            });
        }

        // Extract token
        const token = authHeader.startsWith('Bearer ') ? 
            authHeader.slice(7) : authHeader;

        console.log('Received token:', token);

        if (!token) {
            return res.status(401).json({
                success: false,
                error: 'Invalid token format'
            });
        }

        // Verify token
        const decoded = jwt.verify(token, process.env.JWT_SECRET);

        // Get user data from database
        const [users] = await pool.execute(
            `SELECT 
                id_serenity,
                username,
                status_akun,
                premium_label,
                premium_expires_at,
                extension_token,
                remember_me,
                last_login,
                session_token
            FROM users 
            WHERE id_serenity = ?`,
            [decoded.userId]
        );

        if (users.length === 0) {
            return res.status(401).json({
                success: false,
                error: 'User not found'
            });
        }

        const user = users[0];

        // Validasi session token
        if (user.session_token !== token) {
            return res.status(401).json({
                success: false,
                error: 'Session token tidak valid',
                force_logout: true
            });
        }

        // Check if token matches stored token
        const [sessions] = await pool.execute(
            'SELECT * FROM active_session WHERE id_serenity = ? AND session_token = ?',
            [decoded.userId, token]
        );

        // Add user data to request
        req.user = user;
        req.session = sessions.length > 0 ? sessions[0] : null;

        next();
    } catch (error) {
        console.error('Auth middleware error:', error);
        return res.status(401).json({
            success: false,
            error: 'Invalid token'
        });
    }
};

module.exports = authMiddleware; 