const jwt = require('jsonwebtoken');
const pool = require('../config/database');

const extensionAuthMiddleware = async (req, res, next) => {
    try {
        // Cek format token
        const authHeader = req.headers.authorization;
        if (!authHeader) {
            console.log('No authorization header');
            return res.status(401).json({ error: 'Unauthorized - No token' });
        }

        // Extract token
        const token = authHeader.startsWith('Bearer ') ? 
            authHeader.slice(7) : authHeader;

        if (!token) {
            console.log('No token found in auth header');
            return res.status(401).json({ error: 'Unauthorized - Invalid token format' });
        }

        try {
            // Verifikasi JWT
            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            
            // Debug log
            console.log('Extension token verified:', decoded);

            // Cek token di tabel users dan extension_settings
            const [users] = await pool.execute(
                `SELECT u.id_serenity, u.premium_label 
                FROM users u 
                JOIN extension_settings e ON e.id_serenity = u.id_serenity 
                WHERE e.extension_token = ? AND e.is_active = TRUE`,
                [token]
            );

            if (users.length === 0) {
                console.log('No user found with this extension token');
                return res.status(401).json({ error: 'Invalid session - User not found or token mismatch' });
            }

            // Tambahkan user ID dan status premium ke request
            req.userId = users[0].id_serenity;
            req.isPremium = users[0].premium_label === 1;
            next();
        } catch (jwtError) {
            console.log('JWT verification failed:', jwtError.message);
            return res.status(401).json({ error: 'Unauthorized - Invalid token' });
        }
    } catch (error) {
        console.error('Extension auth middleware error:', error);
        res.status(500).json({ error: 'Internal server error' });
    }
};

module.exports = extensionAuthMiddleware; 